#
# generate ISC DHCPd dhcpd.conf files
#
# author: Alexander Schreiber <als@thangorodrim.de>
#
# version: $Id: module_isc_dhcpd.pm,v 1.7 2002/09/15 22:56:55 als Exp $
#

#
# scass.fes - The backendsystem of SCASS (System Configuration and
#             Administration Support System)
#
# Copyright (C) 2002  Alexander Schreiber <als@thangorodrim.de>
#
# This program is free software; you can redistribute it and/or modify 
# it under the terms of the GNU General Public License as published by 
# the Free Software Foundation; either version 2 of the License, or (at
# your option) any later version.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranty of  
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
# Public License for more details.
#

use strict;

sub dump_net_file_isc_dhcp {
# generates DHCP server config files for ISC DHCP server from the data
# for the net identified by $net_id in the scass database, saves the old
# ones
#
# parameters: $dbh => database handle
#             $net_id => id of the DHCP network to dump
#
# returns: (0, '') on success, (-1, $message) in case of failure
#
# required scass_config parameters: DHCPdConffileDir

    my $dbh = shift;
    my $net_id = shift;


    my $work;

    my $dhcp_file;
    my ($dhcp_filename, $dhcp_filename_base, $old_filename, $old_renamed);
    my ($time, $time_stamp);
    my ($subnet_mask, $default_lease_time, $max_lease_time, $ntp_servers,
        $routers, $domain_name, $domain_name_server, $comment, $last_update,
        $net_name);

    my ($error, $failure);

    my ($option, $value, $key);
    my %options = ( 'subnet_mask' => 'subnet-mask',
                    'default_lease_time' => 'default-lease-time',
                    'max_lease_time' => 'max-lease-time',
                    'domain_name' => 'domain-name',
                    'domain_name_server' => 'domain-name-servers');

    my %options_flags = ('subnet_mask' => 'number',
                         'default_lease_time' => 'number',
                         'max_lease_time' => 'number',
                         'domain_name' => 'string',
                         'domain_name_server' => 'string');

    my %range = ('net' => 'net name',
                 'subnet' => 'sub-net',
                 'netmask' => 'netmask',
                 'range_from' => 'range',
                 'range_to' => '',
                 'broadcast' => 'broadcast-address',
                 'routers' => 'routers',
                 'comment' => 'comment ',
                 'last_update' => 'last update');

    my %host = ( 'net' => 'net name',
                 'hostname' => 'host',
                 'ethernet_address' => 'hardware ethernet',
                 'fixed_address' => 'fixed-address',
                 'comment' => 'comment',
                 'last_update' => 'last update');

    my %range_data;
    my %host_data;
    my %option_data;

    my %config;
    my ($sql, $sth, $data);

# sanity checks
    if ( $net_id eq '' ) {
        return (-1, 'no net_id given');
    }

# grab current time
    $time = time;
    $time_stamp = &get_ISO8601_timestamp($time);

# get current config from db
    %config = &get_config_from_db($dbh);

# build coment header

    $dhcp_file  = "# ISC DHCP server dhcpd.conf\n";
    $dhcp_file .= "#\n# AUTOGENERATED BY SCASS - DO NOT EDIT!\n#\n";
    $dhcp_file .= "# Any changes will be overwritten, change the data in\n";
    $dhcp_file .= "# the SCASS database instead.\n#\n";
    $dhcp_file .= "# generated on $time_stamp\n#\n\n";

# first, gather the information for the zone header

    $sql = "SELECT * from dhcp_net where id = $net_id";
    $sth = $dbh->prepare($sql);
    $sth->execute;
    $data = $sth->fetchrow_hashref;
    $dhcp_filename_base = $data->{net_name};
    foreach $option ( keys(%options) ) {
        $option_data{$option} = $data->{$option};
    }

    $net_name = $data->{net_name};
    $subnet_mask = $data->{subnet_mask};
    $default_lease_time = $data->{default_lease_time};
    $max_lease_time = $data->{max_lease_time};
    $ntp_servers = $data->{ntp_servers};
    $routers = $data->{routers};
    $domain_name = $data->{domain_name};
    $domain_name_server = $data->{domain_name_server};
    $comment = $data->{comment};
    $last_update = $data->{last_update};

# build more comment header
    $dhcp_file .= "# network name: $net_name\n";
    $comment =~ s/\n/\n#          /g;
    $dhcp_file .= "# comment: $comment\n";
    $dhcp_file .= "# last update (database): $last_update\n";
    $dhcp_file .= "#\n\n\n";

# build common block
    foreach $option ( keys(%options) ) {
        if ( $option_data{$option} ne '' ) {
            if ( $options_flags{$option} eq 'number' ) {
                $dhcp_file .= "$options{$option} = $option_data{$option};\n";
            }
            if ( $options_flags{$option} eq 'string' ) {
                $dhcp_file .= "$options{$option} = ";
                $dhcp_file .= "\"$option_data{$option}\";\n";
            }
        }
    }

    $dhcp_file .= "\n\n\n";


# gather range data
    $sql  = "SELECT subnet, netmask(subnet), broadcast(subnet), range_from, ";
    $sql .= "range_to, routers, comment, last_update from dhcp_range ";
    $sql .= "WHERE net = $net_id";

    $sth = $dbh->prepare($sql);
    $sth->execute;
    while ( $data = $sth->fetchrow_hashref ) {
        foreach  $key ( keys(%range) ) {
            $range_data{$key} = $data->{$key};
        }
# data massaging
        $work = $range_data{'comment'};
        $work =~ s/\n/\n#            /g;
        $range_data{'comment'} = $work;
        $work = $range_data{'subnet'};
        $work =~ s/\/.*$//g;
        $range_data{'subnet'} = $work;
        $work = $range_data{'broadcast'};
        $work =~ s/\/.*$//g;
        $range_data{'broadcast'} = $work;

# build range block
        $dhcp_file .= "subnet $range_data{'subnet'} ";
        $dhcp_file .= "netmask $range_data{'netmask'} {\n";
        $dhcp_file .= "#   comment: $range_data{'comment'}\n";
        $dhcp_file .= "#   last update (database): $last_update\n";
        $dhcp_file .= "    range $range_data{'range_from'} ";
        $dhcp_file .= "$range_data{'range_from'};\n";
        $dhcp_file .= "    option broadcast-address ";
        $dhcp_file .= "$range_data{'broadcast'};\n";
        if ( $range_data{'routers'} ne '' ) {
            $dhcp_file .= "    option routers $range_data{'routers'};\n";
        }
        $dhcp_file .= "}\n\n";
    }

    $dhcp_file .= "\n\n\n";

# gather host data
    $sql = "SELECT * FROM dhcp_host WHERE net = $net_id";
    $sth = $dbh->prepare($sql);
    $sth->execute;
    while ( $data = $sth->fetchrow_hashref ) {
        foreach  $key ( keys(%host) ) {
            $host_data{$key} = $data->{$key};
        }
# data massaging
        $work = $host_data{'comment'};
        $work =~ s/\n/\n#            /g;
        $host_data{'comment'} = $work;

# build host block
        $dhcp_file .= "host $host_data{'hostname'} {\n";
        $dhcp_file .= "#   comment: $host_data{'comment'}\n";
        $dhcp_file .= "#   last update (database): $last_update\n";
        $dhcp_file .= "    hardware ethernet ";
        $dhcp_file .= "$host_data{'ethernet_address'};\n";
        $dhcp_file .= "    fixed-address ";
        $dhcp_file .= "$host_data{'fixed_address'};\n";
        $dhcp_file .= "}\n\n";

    }

# prepare dumping the file
    $dhcp_filename  = $config{'DHCPdConffileDir'};
    $dhcp_filename .= "/$dhcp_filename_base";
    $dhcp_filename .= '.conf';

    print "dhcp_filename = |$dhcp_filename|\n";

# save old file, if it exists
    $old_renamed = 0;
    if ( -f $dhcp_filename ) {
        $old_filename = "$dhcp_filename.$time_stamp";
        unless ( rename($dhcp_filename, $old_filename) ) {
# failed ... return failure code together with message
            $error = "failed to rename old $dhcp_filename to $old_filename";
            return (-1, $error);
        }
        $old_renamed = 1;
    }
    $failure = 0;
    open(DHCP, ">$dhcp_filename") or $failure = -1;
    if ( $failure == -1 ) {
        $error = "Failed to open $dhcp_filename : $!";
        if ( $old_renamed == 1 ) {
            unless ( rename($old_filename, $dhcp_filename) ) {
                $error .= " failed to rename $old_filename back to ";
                $error .= "$dhcp_filename";
            }
        }
#        print STDERR "$error\n";
        return (-1, $error);
    }
    print DHCP $dhcp_file;
    print DHCP "\n\n";
    close DHCP;

# ok, everything worked so far - update the database

    $time_stamp = &get_sql_timestamp($time);
    $sql = "UPDATE dhcp_net SET do_commit = false, last_commit = ";
    $sql .= "'$time_stamp' where id = $net_id";
    $sth = $dbh->prepare($sql);
    $sth->execute;

    return (0, '');


#    print $dhcp_file;
}



sub reload_isc_dhcpd {
# reloads ISC dhcpd
# parameter: $dbh -> database handle
# returns: (0, '') on success, (-1, $message) on failure

    my $dbh = shift;

    my %config;
    my $command;
    my $failure;
    my $message;
    my ($line, $output);

    %config = &get_config_from_db($dbh);

    $command = $config{'ISCDHCPdReloadCommand'};
    $failure = 0;
    open(PIPE, "$command|") or $failure = -1;
    if ( $failure == -1 ) {
        $message = "failed to reload ISC dhcpd $!";
#        print STDERR "$message\n";
        return (-1, $message);
    }
    $output = '';
    while ( $line = <PIPE> ) { # there shouldn't be anything to read
        $output .= $line;
    }
    if ( $output ne '' ) {  # any output at all = something bad happend
        $message = "probem reloading ISC dhcpd $output";
#        print STDERR "$message\n";
        return (-1, $message);
    }

    close(PIPE);

    return (0, '');
}










1;
