# generate BIND zonefiles
#
# currently, only supports BIND 9
#
# Author: Alexander Schreiber
#
# Version: $Id: module_bind.pm,v 1.12 2002/09/15 22:56:55 als Exp $
#
#
# scass.bes - The backendsystem of SCASS (System Configuration and
#             Administration Support System)
#
# Copyright (C) 2002  Alexander Schreiber <als@thangorodrim.de>
#
# This program is free software; you can redistribute it and/or modify 
# it under the terms of the GNU General Public License as published by 
# the Free Software Foundation; either version 2 of the License, or (at
# your option) any later version.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranty of  
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
# Public License for more details.
#
#

sub dump_zonefile_bind {
# generates zonefiles for bind - currently bind 9 only - from the zonedate
# for the give zone (via id) in the scass database, saves the old ones.
# parameters: $dbh -> database handle, $zone_id
# returns: (0, '') on success, (-1, $message) in case of failure
#
# required scass_config parameters: DNSBindZoneDir

    my $dbh = shift;
    my $zone_id = shift;

# We build the two zonefiles (forward and reverse) in memory, then rename 
# the old zonefiles to $file_name.$TIMESTAMP and dump the new zonefiles
# in place. Forward file: $zone_name.forward, reverse file: $zone_name.reverse
# This script might fuck up its backup concept if it runs more than once 
# within the same second, therefore, we make damn sure this doesn't happen
# and sleep for a second on startup of this sub.
# This won't help you if you fire of two (or more) copies of this script at
# the same time - but then you're practically begging for it.

    my $forward = '';
    my $reverse = '';

    my $sql;
    my $sth;
    my $work;
    my $data;
    my $time;
    my $time_stamp;
    my @work;
    my $error;
    my $failure;

    my %config;

    my ($forward_file, $reverse_file);
    my ($old_forward_file, $old_reverse_file);
    my ($zone_file_base, $zone_file);
    my ($serial_day, $serial_number);
    my ($domain_name, $serial, $last_commit, $refresh, $retry, $expire,
        $minimum, $zone_admin, $comment, $primary_ns, $network);
    my ($type, $weight, $value);
    my ($ip, $hostname, $hinfo, $txt);
    my ($alias, $target);

# sanity checks
    if ( $zone_id eq '' ) {
#        print STDERR "dump_zonefile_bind(): no zone_id given, skipping";
        return (-1, 'no zone_id given');
    }

# initial sleep - see above
    sleep(1);

# grab current time
    $time = time;

# get current config from db
    %config = &get_config_from_db($dbh);

# first, gather the information for the zone header

    $sql = "SELECT * from dns_zone where id = $zone_id";
    $sth = $dbh->prepare($sql);
    $sth->execute;
    while ( $data = $sth->fetchrow_hashref ) {
        $zone_file_base = $data->{zone_name};
        $domain_name = $data->{domain};
# get old serial and compute new one
        $serial = $data->{serial};
        unless ( $serial =~ /^(\d\d\d\d\d\d\d\d)(\d\d)$/ ) {
            unless ( $serial eq '' ) { # serial not yet set
                print STDERR "serial of zone_id $zone_id in unknown format: ";
                print STDERR "|$serial|, ignoring it\n";
            }
# ok, serial is empty or invalid, so build a new one
            $serial = &get_daystamp($time);
            $serial .= '01';  # first serial for that zone and day
        } else {
# last serial update done today?
            $serial_day = $1;
            $serial_number =$2;
            $work = &get_daystamp($time);
            if ( $serial_number == 99 ) { # maximum reached
#                print STDERR "serial counter too large: $serial_number ";
#                print STDERR "skipping\n";
                return (-1, "serial counter too large");
            }
            if ( $serial_day eq $work ) { # ok, increment
                $serial++;
               } else { # no, its too old, build a new one
                $serial = &get_daystamp($time);
                $serial .= '01';  # first serial for that zone and day
            }
        }
        $zone_admin = $data->{zone_admin};
        $zone_admin =~ s/@/\./;
        $refresh = $data->{refresh};
        $retry = $data->{retry};
        $expire = $data->{expire};
        $minimum = $data->{minimum};
        $primary_ns = $data->{primary_ns};
        $comment = $data->{comment};
        $comment =~ s/\"/ /g;
        $network = $data->{network};
    }

# build zone file headers

# first, forward file head

    $forward = "$domain_name. $minimum  IN SOA $primary_ns. $zone_admin.  (\n";
    $forward .= "\t\t$serial  ; serial\n";
    $forward .= "\t\t$refresh  ; refresh\n";
    $forward .= "\t\t$retry  ; retry\n";
    $forward .= "\t\t$expire  ; expire\n";
    $forward .= "\t\t$minimum ) ; minimum\n\t";
    $forward .= "IN TXT \"$comment\"\n\t";
    $forward .= "IN NS $primary_ns. ;; primary NS\n\n";

    $time_stamp = &get_ISO8601_timestamp($time);
    $forward .= "; autogenerated by SCASS - do not edit!\n";
    $forward .= "; generated: $time_stamp\n\n";

# reverse file head
     
    @work = split(/\./, $network);
    @work = reverse(@work);
    $work = join('.', @work);

    $reverse = "$work.IN-ADDR.ARPA. $minimum IN SOA $primary_ns. ";
    $reverse .= "$zone_admin. (\n";
    $reverse .= "\t\t$serial  ; serial\n";
    $reverse .= "\t\t$refresh  ; refresh\n";
    $reverse .= "\t\t$retry  ; retry\n";
    $reverse .= "\t\t$expire  ; expire\n";
    $reverse .= "\t\t$minimum ) ; minimum\n\t";
    $reverse .= "IN NS $primary_ns. ;; primary NS\n\n";

    $time_stamp = &get_ISO8601_timestamp($time);
    $reverse .= "; autogenerated by SCASS - do not edit!\n";
    $reverse .= "; generated: $time_stamp\n\n";

# get special records
    $sql = "SELECT t.type,s.weight,s.value,s.comment FROM dns_special s, ";
    $sql .= "dns_special_types t WHERE s.zone = 1 AND s.type = t.id";
    $sth = $dbh->prepare($sql);
    $sth->execute;
    while ( $data = $sth->fetchrow_hashref ) {
        $type = $data->{type};
        $value = $data->{value};
        $comment = $data->{comment};
        $comment =~ s/\n/ /g;
        $weight = $data->{weight};
        if ( $type eq 'MX' ) {
            $forward .= "\t";
            $forward .= "IN MX $weight $value. ; $comment\n";
        }
        if ( $type eq 'NS' ) {
            $forward .= "\t";
            $forward .= "IN NS $value. ; $comment\n"; 
        }
    }
    $forward .= "\n\n";

# get host mappings
    $sql = "SELECT ip,hostname,hinfo,txt,comment FROM dns_mapping ";
    $sql .= "WHERE zone = $zone_id";
    $sth = $dbh->prepare($sql);
    $sth->execute;
    while ( $data = $sth->fetchrow_hashref ) {
        $ip = $data->{ip};
        $hostname = $data->{hostname};
        $hinfo = $data->{hinfo};
        $txt = $data->{txt};
        $comment = $data->{comment};
        $comment =~ s/\n/ /g;

        $forward .= "$hostname   IN A $ip ; $comment\n";
        if ( $txt ne '' ) {
            $txt =~ s/\"/ /g;
            $forward .= "            IN TXT \"$txt\"\n";
        }
        if ( $hinfo ne '' ) {
            $hinfo =~ s/\"/ /g;
#            $forward .= "            IN HINFO \"$hinfo\"\n";
        }
        $forward .= "\n";

        $work = $ip;
        $work =~ s/^$network//;
        $work =~ s/^\.//;
        $reverse .= "$work IN PTR $hostname.$domain_name.\n";
        
    }

# get alias mappings
    $forward .= "\n\n";
    $forward .= "; alias records\n\n";
    $sql = "SELECT a.name,m.hostname,a.comment from dns_alias a,";
    $sql .= " dns_mapping m where a.zone = $zone_id and a.target = m.id";
    $sth = $dbh->prepare($sql);
    $sth->execute;
    while ( $data = $sth->fetchrow_hashref ) {
        $alias = $data->{name};
        $target = $data->{hostname};
        $comment = $data->{comment};
        $comment =~ s/\n/ /g;

        $forward .= "$alias  IN CNAME $hostname ; $comment\n";
    }

# ok, file content generated, save old files and dump the new ones in place

#    print "------------------------------------------------------\n";
#    print "\n\n$forward\n\n";
#    print "------------------------------------------------------\n";
#    print "\n\n$reverse\n\n";
#    print "------------------------------------------------------\n";

    $forward_file = "$config{'BindZonefileDir'}/$zone_file_base.forward";
    $reverse_file = "$config{'BindZonefileDir'}/$zone_file_base.reverse";

# if they exist, rename the old files
    if ( -f $forward_file ) {
        $time_stamp = &get_ISO8601_timestamp($time);
        $old_forward_file = "$forward_file.$time_stamp";
        unless ( rename($forward_file, $old_forward_file) ) {
            $error = "failed to rename old $forward_file to ";
            $error .= "$old_forward_file";
#            print STDERR "$error\n";
            return (-1, $error);
        }
    } else {
        $old_forward_file = $forward_file;
        $old_forward_file .= ".old";
    }
    if ( -f $reverse_file ) {
        $time_stamp = &get_ISO8601_timestamp($time);
        $old_reverse_file = "$reverse_file.$time_stamp";
        unless ( rename($reverse_file, $old_reverse_file) ) { 
            $error = "failed to rename old $reverse_file to $old_reverse_file";
#            print STDERR "$error\n";
# damn, re-rename the old renamed forward file
            unless ( rename($old_reverse_file, $forward_file) ) {
# crap, double failure
                $error = "failed to rename old $reverse_file to ";
                $error .="$old_reverse_file, ";
                $error .= "errorhandler to rename $old_reverse_file back to ";
                $error .= "$forward_file failed too";
#                print STDERR "$error\n";
            }
            return (-1, $error);
        }
    }
# start dumping
    $failure = 0;
    open(FORWARD, ">$forward_file") or $failure = -1;
    if ( $failure == -1 ) {
        $error = "Failed to open $forward_file : $!";
        unless ( rename($old_forward_file, $forward_file) ) {
            $error .= " failed to rename $old_forward_file back to ";
            $error .= "$forward_file";
        }
#        print STDERR "$error\n";
        return (-1, $error);
    }
    print FORWARD $forward;
    print FORWARD "\n\n";
    close FORWARD;

    $failure = 0; 
    open(REVERSE, ">$reverse_file") or $failure = -1;
    if ( $failure == -1 ) {
        $error = "Failed to open $reverse_file: $!";
        unless ( rename($old_reverse_file, $reverse_file) ) {
            $error .= " failed to rename $old_reverse_file back to ";
            $error .= "$reverse_file";
        }
# also handle the forward file
        unless ( rename($old_forward_file, $forward_file) ) {
            $error .= " failed to rename $old_forward_file back to ";
            $error .= "$forward_file";
        }
#        print STDERR "$error\n";
        return (-1, $error);
    }
    print REVERSE $reverse;
    print REVERSE "\n\n";
    close REVERSE;

# ok, everything worked so far - update the database

    $time_stamp = &get_sql_timestamp($time);
    $sql = "UPDATE dns_zone SET do_commit = false, last_commit = ";
    $sql .= "'$time_stamp', serial = '$serial' where id = $zone_id";
    $sth = $dbh->prepare($sql);
    $sth->execute;

    return (0, '');

}


sub reload_bind {
# reloads BIND
# parameter: $dbh -> database handle
# returns: (0, '') on success, (-1, $message) on failure

    my $dbh = shift;

    my %config;
    my $command;
    my $failure;
    my $message;
    my ($line, $output);

    %config = &get_config_from_db($dbh);

    $command = $config{'BindReloadCommand'};
    $failure = 0;
    open(PIPE, "$command|") or $failure = -1;
    if ( $failure == -1 ) {
        $message = "failed to reload BIND: $!";
#        print STDERR "$message\n";
        return (-1, $message);
    }
    $output = '';
    while ( $line = <PIPE> ) { # there shouldn't be anything to read
        $output .= $line;
    }
    if ( $output ne '' ) {  # any output at all = something bad happend
        $message = "probem reloading bind: $output";
#        print STDERR "$message\n";
        return (-1, $message);
    }

    close(PIPE);

    return (0, '');    
}

sub build_zone_entry {
# builds named.conf entry with the data given
# parameters: $domain => domain name, $zone_name => name of zone, 
#             $network => network for this domain, 
#             $zone_file_dir => directory where the zonefiles are
# returns: named.conf entry for domain

    my $domain = shift;
    my $zone_name = shift;
    my $network = shift;
    my $zone_file_dir = shift;

    my $named_data = '';

    $named_data .= "\n";
    $named_data .= "// entry for zone $zone_name\n\n";
# entry for the forward file
    $named_data .= "zone \"$domain\" {\n";
    $named_data .= "    type master;\n";
    $named_data .= "    file \"$zone_file_dir/$zone_name.forward\";\n";
    $named_data .= "};\n\n";

# entry for reverse file
    @work = split(/\./, $network);
    @work = reverse(@work);
    $work = join('.', @work);
    $named_data .= "zone \"$work.IN-ADDR.ARPA\" IN {\n";
    $named_data .= "    type master;\n";
    $named_data .= "    file \"$zone_file_dir/$zone_name.reverse\";\n";
    $named_data .= "};\n\n";

    return $named_data;
}

sub write_named_config_include {
# writes the include file for named.conf
# parameters: $dbh => database handle, @zones => ids of zones to include
# returns: (0, '') in case of success, (-1, $message) on failure

    my $dbh = shift;
    my @zones = @_;

    my $time;
    my %config;
    my $time_stamp;
    my ($named_file, $old_named_file);
    my $named_data = '';
    my ($sql, $sth, $data);
    my ($domain, $zone_name, $network);
    my $zone_file_dir;
    my ($work, @work);
    my $zone_id;
    my $error;
    my $failure;



    $time = time();
    $time_stamp = &get_ISO8601_timestamp($time);

    %config = &get_config_from_db($dbh);

    $named_file = $config{'NamedIncludeFile'};
    $old_named_file = "$named_file.$time_stamp";
    $zone_file_dir = $config{'BindZonefileDir'};

    $named_data = "// autogenerated by SCASS - do not edit! \n";
    $named_data .= "// generated: $time_stamp\n\n";

    if ( scalar(@zones) == 0 ) { # no zones given, process all
        $sql = "select zone_name,domain,network from dns_zone";
        $sth = $dbh->prepare($sql);
        $sth->execute;
        while ( $data = $sth->fetchrow_hashref ) {
            $domain = $data->{domain};
            $zone_name = $data->{zone_name};
            $network = $data->{network};
            $named_data .= &build_zone_entry($domain, $zone_name, $network,
                                             $zone_file_dir);
        }

    } else {  # process listed zones
        foreach $zone_id ( @zones ) {
            $sql = "select zone_name,domain,network from dns_zone ";
            $sql .= "where id = $zone_id";
            $sth = $dbh->prepare($sql);
            $sth->execute;
            while ( $data = $sth->fetchrow_hashref ) {
                $domain = $data->{domain};
                $zone_name = $data->{zone_name};
                $network = $data->{network};
                $named_data .= &build_zone_entry($domain, $zone_name, $network,
                                                 $zone_file_dir);
            }
        }
    }

    if ( -f $named_file ) { # don't try no rename a nonexisting file
        unless ( rename($named_file, $old_named_file) ) { # oops
            $error = "failed to rename old $named_file to $old_named_file";
            return (-1, $error);
        }
    }
    $failure = 0;
    open(NAMED_FILE, ">$named_file") or $failure = -1;
    if ( $failure == -1 ) {
        $error = "failed to open named include file $named_file: $!";
# put the old one back into place
        unless ( rename($old_named_file, $named_file) ) { # oh shit
            $error .= "\n failed to rename the old file back!";
        }
        return (-1, $error);
    }
    print NAMED_FILE "$named_data\n";
    close(NAMED_FILE); # #FIXME# we are optimistic here

    return (0, '');
}

1;
