-- stored procedures uses by SCASS
-- author: Alexander Schreiber <als@thangorodrim.de>
-- version: $Id: stored_procedures.sql,v 1.8 2002/09/15 22:56:55 als Exp $


--
-- scass.bes - The backendsystem of SCASS (System Configuration and
--             Administration Support System)
--
-- Copyright (C) 2002  Alexander Schreiber <als@thangorodrim.de>
--
-- This program is free software; you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation; either version 2 of the License, or (at
-- your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but 
-- WITHOUT ANY WARRANTY; without even the implied warranty of  
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details.
--



-- first, all the functions we rely on later

CREATE FUNCTION set_last_update() RETURNS opaque AS '
BEGIN
    new.last_update = current_timestamp;
    return new;
END;'
LANGUAGE 'plpgsql';

CREATE FUNCTION check_zone_name () RETURNS opaque AS '
BEGIN
    IF NEW.zone_name !~* ''^([A-Za-z0-9_.-]+)$'' THEN
        RAISE EXCEPTION ''invalid chars in zone_name'';
    END IF;
    RETURN NEW;
END;'
LANGUAGE 'plpgsql';


CREATE FUNCTION is_valid_hostname (varchar) RETURNS boolean AS '
    my $to_check = shift;
    my @parts;
    my $hostname;
    my $valid;
    my $work;
    my $valid = 1;

    if ( $to_check =~ /\\.\\./ ) {
        $valid = 0;
    }
    if ( $to_check =~ /\\.$/ ) {
        $to_check =~ s/\\.$//;
    }
    $work = length($to_check);
    if ( $work == 0 ) {
        $valid = 0;
    }
    @parts = split(/\\./, $to_check);
    foreach $hostname ( @parts ) {
        $work = length($hostname);
        if ( $work == 0 ) {
            $valid = 0;
        }
        if ( $hostname =~ /^.{0}$/ ) {
            $valid = 0;
        }
        unless ( $hostname =~ /^[a-z]/i ) {
            $valid = 0;
        }
        unless ( $hostname =~ /^([a-z]|[0-9]|\\.|-)+$/i ) {
            $valid = 0;
        }
        $work = chop($hostname);
        if ( $work =~ /^-$/ ) {
            $valid = 0;
        }
    }
    if ( $valid == 1 ) {
        return true;
    } else {
        return false;
    }
;'
LANGUAGE 'plperl';


CREATE FUNCTION check_hostname () RETURNS opaque AS '
BEGIN
    IF is_valid_hostname(NEW.hostname) = false THEN
        RAISE EXCEPTION ''invalid hostname'';
    END IF;
    RETURN NEW;
END;'
LANGUAGE 'plpgsql';

CREATE FUNCTION check_host_name () RETURNS opaque AS '
BEGIN
    IF is_valid_hostname(NEW.name) = false THEN
        RAISE EXCEPTION ''invalid hostname'';
    END IF;
    RETURN NEW;
END;'
LANGUAGE 'plpgsql';

CREATE FUNCTION handle_add_installation () RETURNS opaque AS '
BEGIN
    IF ( NEW.count < 0 ) THEN
        RAISE EXCEPTION ''cannot install negative amount of licenses'';
    END IF;
    PERFORM ''update lm_software set installed_licenses = lm_software.installed_licenses + NEW.count where lm_software.id = NEW.software'';
    return new;
END;'
LANGUAGE 'plpgsql';

CREATE FUNCTION handle_delete_installation () RETURNS opaque AS '
BEGIN
    PERFORM ''update lm_software set installed_licenses = lm_software.installed_licenses - OLD.count where lm_software.id = OLD.software'';
    return old;
END;'
LANGUAGE 'plpgsql';

CREATE FUNCTION handle_update_installation () RETURNS opaque AS '
BEGIN
    IF ( NEW.count < 0 ) THEN
        RAISE EXCEPTION ''cannot install negative amount of licenses'';
    END IF;
        PERFORM ''update lm_software set installed_licenses = lm_software.installed_licenses - OLD.count where lm_software.id = OLD.software'';
        PERFORM ''update lm_software set installed_licenses = lm_software.installed_licenses + NEW.count where lm_software.id = NEW.software'';
 return old;
END;'
LANGUAGE 'plpgsql';


-- stored procedures for the dhcp subsystem

-- check for overlapping range entries the same net id
CREATE FUNCTION check_range_collision () RETURNS opaque AS '
DECLARE collisions INTEGER DEFAULT 0;
BEGIN
    SELECT INTO collisions count(*) FROM dhcp_range WHERE 
    (
        ( 
            range_from > new.range_from 
                OR  
            range_from = new.range_from 
        ) 
            AND 
        ( 
            range_from < range_to 
            OR range_from = new.range_to 
        ) 
            AND 
        net = new.net and id != new.id
    );
    IF collisions != 0 THEN
        RAISE EXCEPTION ''dhpc range collision found'';
    END IF;

    return new;
END;'
LANGUAGE 'plpgsql';

-- check for conflicting IP entries within the same net
CREATE FUNCTION check_ip_collision () RETURNS opaque AS '
DECLARE collisions INTEGER DEFAULT 0;
BEGIN
    SELECT INTO collisions count(*) FROM dhcp_host WHERE 
    (
        ethernet_address = new.ethernet_address 
            AND 
        fixed_address = new.fixed_address 
            AND net = new.net
    );
    IF collisions != 0 THEN
        RAISE EXCEPTION ''colliding host entries found'';
    END IF;

    return new;
END;'
LANGUAGE 'plpgsql';

-- check for IP entries colliding with range entries within the same net
-- works only if fixed_address is specified as IP address rather than FQDN
-- since normally fixed_address is entered as FQDN, this check only really makes
-- sense if we link the dhcp module together with the dns module
CREATE FUNCTION check_ip_range_collision () RETURNS opaque AS '
DECLARE collisions INTEGER DEFAULT 0;
BEGIN
    SELECT INTO collisions count (*) FROM dhcp_range WHERE 
    (
        ( 
            (range_from = new.fixed_address ) 
                OR 
            (range_to = new.fixed_address) 
                OR 
            (
                (range_from < new.fixed_address) 
                    AND 
                (range_to > new.fixed_address)
            )
        ) 
        AND net = new.net
    );
    IF collisions != 0 THEN
        RAISE EXCEPTION ''new host entries colliding with range entries'';
    END IF;

    return new;
END;'
LANGUAGE 'plpgsql';


-- check for range entries colliding with IP entries within the same net
-- works only if fixed_address is specified as IP address rather than FQDN
-- since normally fixed_address is entered as FQDN, this check only really makes
-- sense if we link the dhcp module together with the dns module
CREATE FUNCTION check_range_ip_collision () RETURNS opaque AS '
DECLARE collisions INTEGER DEFAULT 0;
BEGIN
    SELECT INTO collisions count (*) FROM dhcp_host WHERE 
    ( 
        ( 
            (new.range_from = fixed_address ) 
                OR  
            (new.range_to = fixed_address) 
                OR  
            (
                (new.range_from < fixed_address) 
                    AND 
                (new.range_to > fixed_address)
            ) 
        ) 
        AND net = 1
    );
    IF collisions != 0 THEN
        RAISE EXCEPTION ''new range entries colliding with host entries'';
    END IF;

    return new;
END;'
LANGUAGE 'plpgsql';


