/**************************************************************************
 *                                                                        *
 * read.c - program to read textfiles page by page.                       *
 *                                                                        *
 * Planned user interface:                                                *
 *  - 23 lines of text                                                    *
 *  - one status line with short keyhelp                                  *
 *                                                                        *
 * command keys:                                                          *
 *  - q == quit                                                           *
 *  - <space>/+ == one page (21 lines) forward                            *
 *  - b/- == one page (20 lines) backward                                 *
 *  - s == save current position as bookmark                              *
 *  - j == jump to bookmark                                               *
 *  - Ctrl-Z == spawn subshell                                            *
 *                                                                        *
 * Author: Alexander Schreiber <als@thangorodrim.de>                      *
 *                                                                        *
 * Copyright (c) 2001-2003 by Alexander Schreiber                         *
 *                                                                        *
 **************************************************************************
 *                                                                        *
 * Thanks to Gilles for the PAL (Palm Application Library!)               *
 * http://ourworld.compuserve.com/homepages/gilles/                       *
 *                                                                        *
 * It is used in this program in several places.                          *
 *                                                                        *
 **************************************************************************
 *                                                                        *
 * Copyright (C) 2002  Alexander Schreiber <als@thangorodrim.de>          *
 *                                                                        *
 * This program is free software; you can redistribute it and/or modify   *
 * it under the terms of the GNU General Public License as published by   *
 * the Free Software Foundation; either version 2 of the License, or (at  *
 * your option) any later version.                                        *
 *                                                                        *
 * This program is distributed in the hope that it will be useful, but    *
 * WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU       *
 * General Public License for more details.                               *
 *                                                                        *
 * You should have received a copy of the GNU General Public License      *
 * along with this program; if not, write to the                          *
 * Free Software Foundation, Inc.,                                        *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.                *
 **************************************************************************/


#include <stdio.h>
#include <string.h>
#include <sys\stat.h>
#include <conio.h>
#include <process.h>

/*
#include "exec.h"
*/

#include <pal.h>

#define PAGESIZE 23
#define MAXLINELEN 80
#define BUFFERSIZE 4096
#define FONTNAME "smxn0608.hfn"
#define MAX_PAGE_TABLES 256
#define PAGE_TABLE_SIZE 256
#define DEBUG_LOG "read.dbg"

/*  #define DEBUG */

/* yes, a global variable - but _does_ make things easier */
long int *page_table_index[MAX_PAGE_TABLES];

extern unsigned char FontSmall[];
int FontId = SMALL_FONT;

void *zmalloc(size_t size) {
  void *mem;
  char *zero;
  int i;

  mem = (void *) malloc(size);
  if ( mem != NULL ) {
    zero = (char *) mem;
    for ( i = 0; i < size; i++ ) {
      zero[i] = 0;
    }
#ifdef DEBUG
  }
#else
  } else {
    FatalExit("malloc failed - out of memory", 1);
  }
#endif

  return mem;
}

#ifdef DEBUG

void *dbg_malloc(size_t size, char *tag) {
  void *mem;
  FILE *dbg;

  mem = (void *) zmalloc(size);
  dbg = fopen(DEBUG_LOG, "at");
  if ( dbg == NULL ) {
    FatalExit("Failed to open debug log", 1);
  }
  if ( mem != NULL ) {
    fprintf(dbg, "debug.malloc: success (%d bytes, tag = %s)\n", size, tag);
  } else {
    fprintf(dbg, "debug.malloc: failure (%d bytes, tag = %s)\n", size, tag);
  }
  fclose(dbg);

  return(mem);
}

#endif

void showhelp (void) {
/* display short help about program */
  puts("read v0.6 - Copyright (c) 2001-2003 Alexander Schreiber <als@thangorodrim.de>");
  puts("A small program to comfortably read long textfiles.");
  puts("usage: read textfile.txt");
  puts("navigation keys for the program:");
  puts("  use space/'+' to read the next page or 'b'/'-' for the previous page,");
  puts("  with h or ? you will get the keyboard function help");
  puts("  'g' lets you go to a selected page, Ctrl-Z spawns a subshell and 'q' quits.");
}

int init_gfx(void) {
  if ( ! PalInit(1) ) {
    FatalExit("Init PAL failed", 1);
  }
}

void print_gfx(int linex, int liney, char *string) {

  int len, i;

  len = strlen(string);
  for (i = len - 1; i >= 0; i--) {
    if ( (string[i] == 0x0a) || (string[i] == 0x0d) ) {
      string[i] = ' ';
    }
  }
  TextOut(linex, liney, TXT_RULE, FontId, string);
}


void display_help(void) {
  PALWIN *pwin;
  int text_height, x, y;
  char key;

  pwin = OpenWin(WS_HP200, 140, 15, 500, 180, "keyboard help");
  x = 5;
  y = 5;
/*  text_height = FontHeight(FontId);  */
  text_height = 15;
  WinText(pwin, x, y, "keyboard functions:");
  y += text_height;
  WinText(pwin, x, y, "<space> or <+> == next page");
  y += text_height;
  WinText(pwin, x, y, "<-> or <b>     == previous page");
  y += text_height;
  WinText(pwin, x, y, "<g>            == goto page");
  y += text_height;
  WinText(pwin, x, y, "<h> or <?>     == help");
  y += text_height;
  WinText(pwin, x, y, "<q>            == quit");
  y += text_height;
  WinText(pwin, x, y, "<^Z>           == spawn shell");
  y += text_height;
  WinText(pwin, x, y, "");

  y += 2 * text_height;
  WinText(pwin, x, y, "<press any key ...>");
  key = getch();

  CloseWin(pwin);

}
void addindex(long int page, long int pos) {
/* adds a page and its file position to the index */
  int page_table, table_index;
  long int *table;
  int i;
  long int l;
  char strbuf[256];

  page_table = page / PAGE_TABLE_SIZE;
  table_index  = page % PAGE_TABLE_SIZE;
  
  if ( page_table_index[page_table] == NULL ) {
#ifdef DEBUG
    table = (long int *) dbg_malloc(PAGE_TABLE_SIZE * sizeof(l), "page table");
#else
    table = (long int *) zmalloc(PAGE_TABLE_SIZE * sizeof(l));
#endif
    if  (table != NULL ) {
      for ( i = 0; i < PAGE_TABLE_SIZE; i++ ) {
        table[i] = -1;
      }
      page_table_index[page_table] = table;
    }
  } else {
    table = page_table_index[page_table];
  }

  table[table_index] = pos;
  
}

void initindex(void) {
/* initializes the global list */
  int i;

  for ( i = 0; i < MAX_PAGE_TABLES; i++ ) {
    page_table_index[i] = NULL;
  }
}

long int getpagepos(long int page) {
/* checks index looking for page */
/* returns position of page or -1 if not found */

  long int pos = -1;
  int page_table, table_index;
  long int *table;

  page_table = page / PAGE_TABLE_SIZE;
  table_index = page % PAGE_TABLE_SIZE;

  if ( page_table_index[page_table] == NULL ) {
    pos = -1;
  } else {
    table = page_table_index[page_table];
    pos = table[table_index];
  }

  return pos;
}


int buildpageindex(char *filename) {
/* opens file and builds page index from file */
/* returns 0 in case of success, else -1 */
  FILE *input;
  int count;
  int done;
  long int page = 1;
  long int pos;
  long int lines = 0;
  struct stat fileinfo;
  int percent;
  int old_percent = 0;
  int result;
  METER *progress;
  char strbuf[256];
  char buffer[BUFFERSIZE];
  long int file_size;

  result = stat(filename, &fileinfo);
  if ( result != 0 ) {
    sprintf(strbuf, "cannot stat %s", filename);
    FatalExit(strbuf, 1);
  }
  file_size = fileinfo.st_size;
  input = fopen(filename, "rb");
  if ( input == NULL ) {
    sprintf(strbuf, "failed to open %s\n", filename);
    FatalExit(strbuf, 1);
  }

  SetLightSleep(SLEEP_OFF);
  addindex(1, 0);
  progress = OpenMeter(PROG_METER, 150, 80, 3, 0, 100, "reading file");
  while ( !feof(input) ) {
    for ( count = 0; count < PAGESIZE; count++ ) {
      fgets(buffer, BUFFERSIZE, input);
      lines++;
    }
    page++;
    pos = ftell(input);
    addindex(page, pos);
    percent =  (pos * 100) / file_size;
    if ( percent != old_percent ) {
      UpdateMeter(progress, percent);
      old_percent = percent;
    }
  }

  fclose(input);
  CloseMeter(progress);
  SetLightSleep(SLEEP_ON);

  return 0;
}


int displaytext(char *filename) {
/* displays the file one page at a time */

  int key = 0;
  FILE *input;
  char *line;
  long int lines;
  long int page, lastpage;
  int result;
  struct stat fileinfo;
  long int percent;
  char *display_name;
  int len;
  int lastbs;
  char strbuf[256];
  int linex, liney;
  int text_height;
  long  int pos;
  long int i;
  long int go_page;
  int msg_result;
  int linelen;

  ClrBlock(0, 0, 639, 199, WHITE_COLOR);
  text_height = FontHeight(FontId);
  linex = 0;
  liney = 0;
  len = strlen(filename);
  lastbs = 0;
  for ( i = 0; i < len; i++ ) {
    if ( filename[i] == '\\' ) {
      lastbs = i;
    }
  }
  if ( lastbs != 0 ) {
    lastbs++;
  }
  display_name = filename + lastbs;
#ifdef DEBUG
  line = (char *)dbg_malloc(BUFFERSIZE, "display line buffer");
#else
  line = (char *)zmalloc(BUFFERSIZE);
#endif

  result = stat(filename, &fileinfo);
  if ( result != 0 ) {
    sprintf(strbuf, "cannot stat %s", filename);
    FatalExit(strbuf, 1);
  }
  input = fopen(filename, "rb");
  if ( input == NULL ) {
    sprintf(strbuf, "failed to open %s\n", filename);
    FatalExit(strbuf, 1);
  }

/* find last page */
  lastpage = 1;
  for (i = 1; i < 10000; i++ ) {
    pos = getpagepos(i);
    if ( pos != -1 ) {
      lastpage = i;
    } else {
      i = 10000;
    } 
  }

  page = 1;
  while ( key != 'q' ) {
    ClrBlock(0, 0, 639, 199, WHITE_COLOR);
    liney = 0;
    pos = getpagepos(page);
    fseek(input, pos, SEEK_SET);
    for ( lines = 0; lines < PAGESIZE; lines++ ) {
      line[0] = 0;
      fgets(line, BUFFERSIZE, input);
      line[strlen(line) - 1] = 0;
      /* puts(line); */
      linelen = strlen(line);
      if ( linelen > (MAXLINELEN + 1) ) {
        line[MAXLINELEN - 1] = '>';
        line[MAXLINELEN] = 0;
      }
      print_gfx(linex, liney, line);
      liney += text_height;
    }
    liney += 4;
    Line(0, liney, 639, liney);
    liney += 4;
    percent = (page * 100) / lastpage;
    sprintf(strbuf, 
    "file: %12s --> page %ld/%ld = %3ld%% <--      [+/ /b/-/g/h/?/q/^Z]:   ",
            display_name, page, lastpage, percent);
    print_gfx(linex, liney, strbuf);
    liney += text_height;
    key = getch();
    /*  puts(" ");  */
    switch ( key ) {
      case ' ' : {
             pos = getpagepos(page + 1);
             if ( pos != -1 ) {
               page++;
             } else {
               print_gfx(linex, liney, 
                       "\r --> last page <--                      ");
               liney += text_height;
             }
             break;
           }
  
      case '+' : {
             pos = getpagepos(page + 1);
             if ( pos != -1 ) {
               page++;
             } else {
               print_gfx(linex, liney, 
                       "\r --> last page <--                      ");
               liney += text_height;
             }
             break;
           }
  
      case 'b' : {
             pos = getpagepos(page - 1);
             if ( pos != -1 ) {
               page--;
             } else {
               print_gfx(linex, liney, 
                       "\r --> first page <--                      ");
               liney += text_height;
             }
             break;
           }
  
      case '-' : {
             pos = getpagepos(page - 1);
             if ( pos != -1 ) {
               page--;
             } else {
               print_gfx(linex, liney, 
                       "\r --> first page <--                      ");
               liney += text_height;
             }
             break;
           }
      case 'g' : {
             msg_result = MsgBox("Enter page number to go to", "page number",
                                 strbuf, "!OK   |Cancel");
             if ( msg_result == 0 ) {
               go_page = page;
               sscanf(strbuf, "%ld", &go_page);
               if ( go_page != page ) {
                 page = go_page;
               }
             }
             break;
           }
      case 'h' : {
              display_help();
              break;
           }
      case '?' : {
              display_help();
              break;
           }
      case 26 : { /* Ctrl-Z pressed */
            PalDeInit(1);
            free(line);
            puts("--> spawning command.com <--");
            result = spawnlp(P_WAIT, "command.com", NULL);
            if ( result == -1 ) {
              puts("--> spawning command.com failed <--");
            }
            init_gfx();
#ifdef DEBUG
            line = (char *) dbg_malloc(BUFFERSIZE, "display line buffer (re)");
#else
            line = (char *) zmalloc(BUFFERSIZE);
#endif	

            break;
          }
    }
  }
}

/*
 *int dumplist(void) {
 *  int pages = 0;
 *  pagelist *list;
 *
 *  list = root;
 *
 *  while ( list != NULL ) {
 *    printf("(%ld. %ld)  ", list->page, list->pos);
 *    pages++;
 *    list = list->next;
 *  }
 *
 *}
 */

int main(int argc, char **argv) {

  struct stat fileinfo;

  if ( ! ( ( argv[1] != NULL ) && ( argv[2] == NULL ) ) ) {
  showhelp();
  return(1);
  } else {
    if ( stat(argv[1], &fileinfo) == -1 ) {
      printf("%s", argv[1]);
      perror("");
      showhelp();
      return(2);
    } else {
      init_gfx();
      initindex();
      buildpageindex(argv[1]);
      displaytext(argv[1]);
      PalDeInit(1);
    }
  }
}


