;;;; file: lns.lisp
;;;; author: Alexander Schreiber <als@thangorodrim.de>
;;;; 
;;;; Core of the lisp-network-server - a simple Network SuperServer for
;;;; CommonLisp network applications.
;;;;
;;;; Copyright (C) 2006 Alexander Schreiber
;;;;
;;;; This library is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU Library General Public
;;;; License as published by the Free Software Foundation;
;;;; version 2 of the License.
;;;;
;;;; This library is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;; Library General Public License for more details.
;;;;
;;;; You should have received a copy of the GNU Library General Public
;;;; License along with this library; if not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;;
;;;; author : Alexander Schreiber <als@thangorodrim.de>
;;;; version: $Id: lns.lisp 1707 2007-12-16 23:47:30Z als $
;;;;
;;;;


(defpackage :lisp-network-server
  (:nicknames :lns)
  (:use :common-lisp)
  (:documentation
   "LNS - lisp-network-server is a network superserver for CommonLisp network
applications inspired by the model of the UNIX inetd superserver.")
  (:export #:register-service-handler
           #:unregister-service-handler
           #:get-service-listeners
           #:get-service-handler
           #:get-log-file
           #:set-log-file
           #:set-max-threads
           #:get-max-threads
           #:write-log
           #:valid-ip-addressp
           #:start-framework
           #:daemon
           ))

(require :sb-bsd-sockets)
(require :sb-posix)
; (require :sb-unix)

(in-package :lns)



(proclaim '(optimize (safety 3))) ; just being careful

(defparameter *lns-revision* "$Revision: 1707 $" "revision of the server core")

;;; global runtime config - this might need mutex protection for
;;; runtime updates

;;; note: to avoid races, all accesses to this have to be guarded by
;;; mutex
(defvar *listeners*
  (make-hash-table :test #'equal) "mapping network ports to listeners")

(defvar *listener-mutex* (sb-thread:make-mutex :name "listener config lock"
                                               :value nil)
  "Mutex protecting access to the listener config")


;; adjust this to something sensible by calling (set-log-file ..) with
;; a suitable pathname
(defvar *log-file*
  (make-pathname :name "lns"
                 :type "log")
  "log file for LNS")

(defvar *log-config-mutex* (sb-thread:make-mutex :name "logfile config lock"
                                                 :value nil)
  "Mutex protecting access to the logfile config.")

(defvar *log-file-handle* nil "the open log file")
(defvar *log-file-handle-mutex*
  (sb-thread:make-mutex :name "log file handle lock"
                        :value nil)
  "Mutex to protect it")


(defvar *max-threads* 64 "Maximum number of active threads in image.")

(defvar *thread-config-mutex* (sb-thread:make-mutex :name "thread config lock"
                                                    :value nil)
  "Mutex protecting thread config.")


(defvar *log-writer-active* nil "Do we have a running logwriter?")
(defvar *log-writer-active-mutex*
  (sb-thread:make-mutex :name "log writer state mutex"
                        :value nil)
  "Protect the log writer state.")

(defvar *log-writer-interval* 30 "sleep this many seconds, flush, repeat")
(defvar *log-writer-interval-mutex*
  (sb-thread:make-mutex :name "log writer state mutex"
                        :value nil)
  "Protect the log writer state.")


;;; macros

(defmacro get-listener (address port)
  "Reading values from *listener-mutex* with mutex protection."
  `(sb-thread:with-recursive-lock (*listener-mutex*)
    (gethash (cons ,address ,port) *listeners*)))


(defmacro set-listener (address port handler)
  "Setting values in *listeners* with mutex protection."
  `(sb-thread:with-recursive-lock (*listener-mutex*)
    (setf (gethash (cons ,address ,port) *listeners*) ,handler)))


(defmacro rem-listener (address port)
  "Removing values from *listeners* with mutex protection."
  `(sb-thread:with-recursive-lock (*listener-mutex*)
    (remhash (cons ,address ,port) *listeners*)))
                        

;;; helper

(defun scream-and-die (message)
  "Write message to syslog as LOG_ERR and terminate."
  (ignore-errors ;; we're gonna die anyway
    (sb-posix:syslog sb-posix:log-err
                     (format nil
                             "aborting due to critical error: ~a"
                             message))
    (sb-ext:quit)))


(defun syslog (message)
  "Write message to syslog."
  (sb-posix:syslog sb-posix:log-err message))


;;; implementation code


(defun daemon (&key (standard-input "/dev/null")
               (standard-output "/dev/null")
               (error-output "/dev/null")
               (pid-file "/var/run/lns.pid"))

  "Turn the currently running process into a daemon. This includes
   - fork
   - in parent: exit()
   - setsid()
   - chdir('/')
   - reopen fd 0, 1, 2 to /dev/null
   - write PID to pid-file"

  (syslog "daemonization started")
  (if (not (equal (sb-posix:fork) 0))
      (SB-UNIX:UNIX-EXIT 0)
      (progn
        ;; apparently, while there seems to be a hook to the setsid 
        ;; UNIX function, it is not part of any exported SBCL API
        ;; Oh well, I still need it
        (sb-unix::unix-setsid)
        (SB-POSIX:CHDIR "/")
        (handler-case
            (progn
              (setf *standard-input*
                    (open standard-input
                          :direction :input
                          :if-does-not-exist :error))
              (setf *standard-output*
                    (open standard-output
                          :direction :output
                          :if-does-not-exist :create
                          :if-exists :append))
              (setf *error-output*
                    (open error-output
                          :direction :output
                          :if-does-not-exist :create
                          :if-exists :append))
              (syslog "daemonization complete")
              (with-open-file (stream pid-file :direction :output
                                      :if-exists :overwrite
                                      :if-does-not-exist :create)
                (format stream "~a" (sb-posix:getpid))))
          (sb-int:simple-file-error (message)
            (scream-and-die     
             (format nil "file error: ~a" message)))
          (serious-condition (message)
            (scream-and-die 
             (format nil "error: ~a" message)))))))
 


(defun get-services-listeners ()
  "Return a list of address/port pairs with currently registered handlers."
  (let ((listeners '()))
    (sb-thread:with-recursive-lock (*listener-mutex*)
      (maphash #'(lambda (key value)
                   (declare (ignore value))
                   (setf listeners (append listeners key)))
               *listeners*))
    listeners))

(defun get-service-handler (address port)
  "Return the service handler for the specified address & port,
if one is registered."
  (get-listener address port))


(defun get-named-threads (thread-name)
  "Returns list of threads whose names match the specified thread-name."
  (remove-if-not #'(lambda (item)
                     (string= thread-name
                              (sb-thread:thread-name item)))
                 (sb-thread:list-all-threads)))


(defun make-thread-id (&optional (thread nil))
  "Takes a thread and returns a unique thread id based on the
printed representation of the value. Uses the current-thread if none
is specified."
  (let ((printed (format nil "~a"
                         (if thread
                             thread
                             sb-thread:*current-thread*))))
    (subseq printed
            (1+ (position #\{ printed :from-end t))
            (position #\} printed :from-end t))))


(defun get-iso8601-timestamp (&optional universal-time)
  "Returns an ISO8601 timestamp, using either current time or the
supplied universal time."
  (multiple-value-bind (ss mm hh dd mo yy) ; drop unneeded values
      (decode-universal-time 
       (if (integerp universal-time)
           universal-time
           (get-universal-time)))
    (format nil "~4,'0D-~2,'0D-~2,'0DT~2,'0D:~2,'0D:~2,'0D"
            yy mo dd hh mm ss)))

;;; house-keeping code

(defun write-log (entry)
  "Write the specified (text) entry to the LNS logfile."
  ;; ok, is our log file already open?
  (sb-thread:with-recursive-lock (*log-file-handle-mutex*)
    (if (null *log-file-handle*)
        (setf *log-file-handle*
              (handler-case
                  (open (sb-thread:with-recursive-lock (*log-config-mutex*)
                          *log-file*)
                        :direction :output
                        :external-format :utf-8
                        :if-exists :append
                        :if-does-not-exist :create)
                (serious-condition (condition)
                  (sb-posix:syslog sb-posix:log-err
                                   (format nil
                                           "error opening log file ~a: ~a"
                                           *log-file*
                                           condition))
                  (scream-and-die "failed to open log file")))))
    ;; should be opened and set now
    (handler-case
        (format *log-file-handle*
                "~a thread ~a: ~a~%"
                (get-iso8601-timestamp)
                (make-thread-id)
                entry)
      (serious-condition (condition)
        (sb-posix:syslog sb-posix:log-err
                         (format nil
                                 "failed to write log file: ~a"
                                 condition))
        (scream-and-die "log write failure")))))



(defun get-log-file ()
  "Returns the currently configured logfile."
  (sb-thread:with-recursive-lock (*log-config-mutex*)
    *log-file*))


(defun set-log-file (path &key (logchange t))
  "Sets the new logfile and logs the change."
  (assert (or
           (pathnamep path)
           (stringp path))
          (path)
          "The logfile path must be a string or a pathname, not ~S." path)
  (assert (typep logchange 'boolean)
          (logchange)
          "logchange must be boolean, not ~S" logchange)

  (let ((old-log-file (get-log-file)))
    (if logchange
        (write-log (format nil
                           "log file switched to ~a"
                           path)))
    (sb-thread:with-recursive-lock (*log-config-mutex*)
      (setf *log-file* path))
    ;; now we switched the log file name, lets also switch files
    (sb-thread:with-recursive-lock (*log-file-handle-mutex*)
      (if (not (null *log-file-handle*))
          (close *log-file-handle*))
      (setf *log-file-handle*
            (open (sb-thread:with-recursive-lock (*log-config-mutex*)
                    *log-file*)
                  :direction :output
                  :if-exists :append
                  :if-does-not-exist :create)))
    (if logchange
        (write-log (format nil
                           "log file switched from ~a to ~a"
                           old-log-file
                           path)))))
  


(defun get-max-threads ()
  "Query the current maximum number of allowed threads."
  (sb-thread:with-recursive-lock (*thread-config-mutex*)
    *max-threads*))


(defun set-max-threads (count)
  "Set the maximum number of allowed threads."
  (assert (and
           (integerp count)
           (< 0 count))
          (count)
          "Maximum number of thread must be a positive integer, not ~S." count)
  (sb-thread:with-recursive-lock (*thread-config-mutex*)
    (setf *max-threads* count)))




(defun too-many-threadsp (&key (wait nil) (wait-time 1))
  "Returns T if there are currently too many (>= *max-threads*) threads
active, NIL otherwise. Will wait once for wait-time seconds and check
again if told so, but will return immediately with results by default."
  (let ((max-threads (sb-thread:with-recursive-lock (*thread-config-mutex*)
                       *max-threads*)))
    
    (if (< max-threads (length (sb-thread:list-all-threads)))
        (if (not (null wait))
            (progn
              (sleep wait-time)
              (if (< max-threads (length (sb-thread:list-all-threads)))
                  t
                  nil))
            t)
        nil)))



(defun make-string-from-inet-address (address)
  "Take a inet address (simple vector 4) and return a string."
  (format nil "~a.~a.~a.~a"
          (elt address 0)
          (elt address 1)
          (elt address 2)
          (elt address 3)))

  

;;; connection handler
(defun handle-connection (link address port)
  "Handle the established network connection.
Basic approach:
 - set up simple condition handlers that attempt to close the network
   connection and then exit the thread
 - call the function that was specified in the global listener table
   with two arguments: a two-way stream (from the socket) and
   the socket peer information
 - attempt to close the socket and exit the thread when the function is done."
  ;; cache some stuff we need a few times (and that won't change)
  (let ((bound-ip nil)
        (bound-port nil)
        (remote-peer-ip nil)
        (remote-peer-port nil)
        (thread-id nil)
        (handler (get-listener address port)))

    ;; init cache values
    (multiple-value-bind (remote-host remote-port)
        (sb-bsd-sockets:socket-peername link)
      (setf remote-peer-ip
            (format nil "~a"
                    (make-string-from-inet-address remote-host)))
      (setf remote-peer-port remote-port))
    (multiple-value-bind (ip port)
        (sb-bsd-sockets:socket-name link)
      (setf bound-ip 
            (format nil "~a"
                    (make-string-from-inet-address ip)))
      (setf bound-port port))

    
    (setf thread-id (make-thread-id))
    
    (write-log (format nil
                     "connection to port ~a:~a from ~a:~a"
                     address
                     port
                     remote-peer-ip
                     remote-peer-port))

    ;; simple overload protection: if there are too many threads already,
    ;; just slam the door shut again

    (if (too-many-threadsp)
        (progn
          (sb-bsd-sockets:socket-close link)
          (write-log
           (format nil "warning: too many threads, connection aborted"))
          (sb-ext:quit)))
    
    (unwind-protect 
         (if (null handler)
             (write-log (format nil
                                "no listener defined for port ~a:~a"
                                address
                                port))
             (let ((stream nil))

               (handler-case
                   (setf
                    stream
                    (sb-bsd-sockets:socket-make-stream link
                                                       :input t
                                                       :output t
                                                       :buffering :full
                                                       :external-format
                                                       :utf-8
                                                       :element-type
                                                       :default))
                 (type-error (condition)
                   (write-log
                    (format nil
                            "stream-create fail on port ~a:~a: ~a"
                            address
                            port
                            condition))
                   (sb-bsd-sockets:socket-close link)
                   (sb-ext:quit)))
               
               (handler-case
                   (progn
                     (write-log
                      (format nil
                              "handoff to ~a" (format nil "~a" handler)))
                     (funcall
                      handler
                      stream
                      bound-ip
                      bound-port
                      remote-peer-ip
                      remote-peer-port))
                 (stream-error (condition)
                   ;; If we get a stream-error, then this usually means that
                   ;; our client dropped the connection. Any attempt to write
                   ;; something down that stream will just fill up the outgoing
                   ;; buffers. And we most likely already have stuff sitting
                   ;; there, waiting to be read by the client. Which is never
                   ;; going to happen so the file descriptor of that stream
                   ;; cannot be gc'ed. Which sucks because we're gonna run out
                   ;; of the little buggers sooner or later and then we won't
                   ;; be able to answer requests anymore.
                   ;; So, to deal with this we have to cut our losses and that
                   ;; means killing the stream real fast and hard, throwing
                   ;; away anything sitting in buffers somewhere.
                   ;; Thank X3J13 for the :abort keyword to close.
                   (progn
                     (write-log
                      (format nil "stream-error in handler: ~a" condition))
                     (close stream :abort t)))
                 (serious-condition (condition)
                   (write-log
                    (format nil "serious-condition in handler: ~a"
                            condition))))))

      (write-log (format nil "handler cleanup"))
      ;; this is ugly, but at that point we don't give 
      ;; a flying fuck about any errors, we just want it to
      ;; be over, quick like
      ;; and besides, thread termination cleans out
      ;; the socket we failed to close properly anyway
      ;;
      ;; the later turned out not to be the case ... so we close it ourselves

      (handler-case
          (sb-bsd-sockets:socket-close link)
        (serious-condition (condition)
          (write-log (format nil
                             "serious condition while closing socket: ~a"
                             condition))
          (handler-case
              (progn
                (sb-bsd-sockets:socket-close link)
                (close link))
            
            (serious-condition (condition)
              (write-log (format nil
                                 "serious condition 2 while closing socket: ~a"
                                 condition))))))
      (sb-ext:quit))))


  


;;; network listener

(defun listen-on-network (socket address port)
  "Listen on the specified network address (string) TCP port for connections."
  (unwind-protect
       (loop
        (sb-bsd-sockets:socket-listen socket 5)
        (let ((link (sb-bsd-sockets:socket-accept socket)))
          (sb-thread:make-thread #'(lambda ()
                                     (handle-connection link address port))
                                 :name (format nil
                                               "connection handler ~a:~a/tcp"
                                               address
                                               port))))
    (sb-bsd-sockets:socket-close socket)))


(defun valid-ip-addressp (address)
  "Predicate to check wether the supplied argument is a valid IP address.
Currently limited to IPv4 which is OK right now since SBCL only supports
IPv4 at the time."
  (if (and (vectorp address)
           (= (length address) 4)
           (reduce #'(lambda (a b)
                       (and a b))
                   (map 'list #'(lambda (item) (if (and
                                                    (< -1 item)
                                                    (> 256 item))
                                            t
                                            nil)) address)))
      t
      nil))



;;; configuration functions

(defun register-service-handler (address port handler)
  "Register a new service handler listening on the specified port and
start the associated network listener."
  ;; first, argument checks
  ;; address must at least look like a valid IP address
  ;; port needs to be in the 1 .. 65535 integer range
  ;; handler needs to be a function

  (assert (valid-ip-addressp (sb-bsd-sockets:make-inet-address address))
          (address)
          "address must be a valid IP address, not ~S." address)
  
  (assert (and (integerp port)
               (< port 65536)
               (> port 0))
          (port)
          "TCP port must be an integer 1 .. 65535, not ~S." port)

  (assert (functionp handler)
          (handler)
          "The port handler must be a function, not ~S." handler)

  ;; if there is already a handler registered do nothing
  (if (null (get-listener address port))
      (progn
        ;; add the handler to the global handler table
        (set-listener address port handler)
        (write-log (format nil
                           "registered handler ~a for ~a:~a/tcp"
                           (format nil "~a" handler)
                           address
                           port))
        ;; initialize the listener thread
        (let ((socket (make-instance 'sb-bsd-sockets:inet-socket
                                     :type :stream
                                     :protocol :tcp)))
          (handler-case
              (sb-bsd-sockets:socket-bind socket
                                          (sb-bsd-sockets:make-inet-address
                                           address)
                                          port)
            (sb-bsd-sockets:socket-error (condition)
              (write-log (format nil
                                 "error binding socket to ~a:~a/tcp: ~a"
                                 address
                                 port
                                 condition))
              (rem-listener address port)
              (write-log (format nil
                                 "errors, unregistered handler for ~a:~a/tcp"
                                 address
                                 port))
              (return-from register-service-handler nil)))
          (setf (sb-bsd-sockets:sockopt-reuse-address socket) t)
          (write-log
           (format nil
                   "thread ~a starting to listen on ~a:~a"
                   (make-thread-id
                    (sb-thread:make-thread #'(lambda ()
                                               (listen-on-network socket
                                                                  address
                                                                  port))
                                           :name
                                           (format nil
                                                   "network listener ~a:~a/tcp"
                                                   address
                                                   port)))
                   address
                   port)))
        t)
      (progn
        (write-log (format nil
                           "error, handler for ~a:~a/tcp already registered"
                           address
                           port))
        nil)))
      
  


(defun unregister-service-handler (address port)
  "Unregister a service handler listening on the specified port and stop
the associated network listener."
  ;; first, argument checks
  ;; port needs to be in the 1 .. 65535 range

  (assert (valid-ip-addressp (sb-bsd-sockets:make-inet-address address))
          (address)
          "address must be a valid IP address, not ~S." address)
  
  (assert (and (integerp port)
               (< port 65536)
               (> port 0))
          (port)
          "TCP port must be an integer 1 .. 65535, not ~S." port)

  ;; remove the handler from the global handler table
  ;; important: to avoid races, first kill the listener thread and
  ;; then remove the handler entry


  (if (null (get-listener address port))
      (write-log
       (format nil
               "trying to unregister non-existing handler for ~a:~a/tcp"
               address
               port))
      (progn
        (ignore-errors  ; brute force and ignorance
          (dolist (item (get-named-threads
                         (format nil
                                 "network listener ~a:~a/tcp"
                                 address
                                 port)))
            (sb-thread:terminate-thread item)
            (write-log (format nil
                               "terminated network listener thread ~a"
                               (make-thread-id item)))))
          

        (rem-listener address port)
        ;; apparently, just closing the socket in Lisp code (which happens
        ;; when we kill the network handler thread) doesn't
        ;; guarantee that this is also told to the OS (and the socket closed
        ;; for real)
        ;; But forcing a (full) gc results in the "zombie" socket being
        ;; collected and the port to be freed - otherwise, doing an
        ;; unregister-service-handler followed by a register-service-handler
        ;; for the same port will most likely fail, which is irritating at best
        ;; this solution is ugly, but it works
        ;; yes, full gc may be expensive, but then, our users hopefully
        ;; aren't replacing their service handlers every 2 seconds ...
        ;; 
        ;; Turns out that even normal full gc doesn't always work. So
        ;; we now do a full gc with generational gc disabled. This one's
        ;; gonna hurt a tad more, but what the hell.
        (sb-ext:gc :gengc nil :full t)
        (write-log (format nil
                           "unregistered handler for ~a:~a/tcp"
                           address
                           port)))))


(defun log-flush ()
  "Flush the lns.log, called by log-writer."
  (let ((log-file-path))
    
    (setf log-file-path
          (sb-thread:with-recursive-lock (*log-config-mutex*)
            *log-file*))
    (sb-thread:with-recursive-lock (*log-file-handle-mutex*)
      (ignore-errors ; not nice, but we (hopefully) can ignore those
        (if (not (null *log-file-handle*))
            (close *log-file-handle*))) ; should force a flush
      ;; always re-open it
      ;; now, if we run into trouble here we are up shit creek without
      ;; a paddle - we can't really afford to drop into the debugger since
      ;; we are supposed to run as a daemon ... hmm
      ;; two ways to handle this:
      ;;  - pretend nothing happened and continue
      ;;  - die horribly
      ;; remember: we can't afford to drop into the debugger and we no
      ;; longer have log channel to weep into either
      (setf *log-file-handle*
            (open log-file-path
                  :direction :output
                  :external-format :utf-8
                  :if-exists :append
                  :if-does-not-exist :create)))))



(defun log-writer ()
  "Our logs are written asynchronously to avoid the logwriter bogging
down request handling. This means that they only get written if we
generate more data than fits whatever buffers are there. To reduce this
effect, we use this log-writer who basically closes/opens the logfile
every *log-writer-interval* seconds."
  ;; if shit hits the fan in here we most likely have no log file
  ;; anymore - which is Bad (TM), so we whine to syslog and die screaming

  (handler-case
      (unwind-protect
           (loop
            (sleep (sb-thread:with-recursive-lock (*log-writer-interval-mutex*)
                     *log-writer-interval*))
            (log-flush))
        (log-flush))  ; that should take care of flush-on-process-exit
    (serious-condition (condition)
      (sb-posix:syslog sb-posix:log-err
                       (format nil
                               "error during log-flush: ~a"
                               condition))
      (sb-posix:syslog sb-posix:log-err
                       "error during log-file flush, aborting")
      (sb-ext:quit))))
  



(defun start-framework ()
  "Do the initial startup for the lisp-network-server framework."
  ;; first, we grab access to syslog so we have a channel to weep into
  ;; in case of Shit Asploding Real Bad (TM)
  (sb-posix:openlog "lisp-network-server" sb-posix:log-pid)
  
  ;; right now, only start the logwriter thread
  (sb-thread:with-mutex (*log-writer-active-mutex* :value t :wait-p t)
    (if (null *log-writer-active*)
        (setf *log-writer-active*
              (sb-thread:make-thread #'(lambda ()
                                         (log-writer))
                                     :name "lns log writer thread")))))
  
