;;;; file: simple-services.lisp
;;;; 
;;;; Partial implementaton of simple TCP services for LNS.
;;;;
;;;; Copyright (C) 2006 Alexander Schreiber
;;;;
;;;; This library is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU Library General Public
;;;; License as published by the Free Software Foundation; either
;;;; version 2 of the License.
;;;;
;;;; This library is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;; Library General Public License for more details.
;;;;
;;;; You should have received a copy of the GNU Library General Public
;;;; License along with this library; if not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;;
;;;; author : Alexander Schreiber <als@thangorodrim.de>
;;;; version: $Id: simple-tcp.lisp 1518 2006-10-26 20:11:53Z als $
;;;;
;;;;


(defpackage :lns.simple-tcp
  (:use :common-lisp)
  (:documentation
   "Partial implementation of simple TCP services for LNS.")
  (:export #:echo-server
           #:daytime-server
           #:fortune-server
           #:time-server
           #:hello-server
           #:add-fortunes
           #:start-simple-tcp-servers
           ))


(in-package :lns.simple-tcp)


;;; tons of data loaded on startup

(defvar *fortunes* nil  "fortune store")

(defvar *fortune-mutex* (sb-thread:make-mutex :name "fortunes lock"
                                              :value nil)
  "Mutex protecting access to the jar of fortune cookies.")

(defvar *fortune-paths* 
  '(#P"/usr/share/games/fortunes/*" 
    #P"/usr/share/fortune/*"
    #P"/usr/share/fortune/cs")
  "Paths to system fortunes, may need tweaking on your system.")


(defvar *address* "0.0.0.0" "default listen address for the services")

;;; ----

  
  

(defun echo-server (stream ip port peer-ip peer-port)
  "Echos everything it reads - simple echo server according to RFC862."
  (declare (ignore ip port peer-ip peer-port))
  (let ((line nil))
    (loop
     (setf line (read-line stream nil nil))
     (if (null line)
         (sb-ext:quit)
         (format stream "~a~%" line)))))

;;; ------

(defmacro month-name (month-number)
  `(nth (1- ,month-number)
    '("Jan" "Feb" "Mar" "Apr" "May" "Jun"
      "Jul" "Aug" "Sep" "Oct" "Nov" "Dec")))

(defmacro day-name (day-number)
  `(nth ,day-number
    '("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun")))

(defun daytime-server (stream ip port peer-ip peer-port)
  "Prints the current local daytime and terminates, see RFC 867."
  (declare (ignore ip port peer-ip peer-port))
  (multiple-value-bind (ss mm hh dd mo yy dow dst tz)
      (decode-universal-time (get-universal-time))
    (format stream
            "~a ~a ~2,' D ~2,'0D:~2,'0D:~2,'0D ~a ~a~%"
            (day-name dow)
            (month-name mo)
            dd
            hh
            mm
            ss
            tz
            yy)))

;;; ----------

(defun word-to-bytesequence (number)
  "convert stuff"
;  (declare (type '(vector unsigned-byte) work))
  (let ((word (make-array 4 :element-type '(unsigned-byte 8)))
        (work-number (rem number (expt 256 4)))) ; enforce number < 2^32
    (dotimes (position 4)
      (let ((byte-num (- 3 position)))
        (multiple-value-bind (quotient remainder)
            (truncate work-number (expt 256 byte-num))
          (setf (aref word position) quotient)
          (setf work-number remainder))))
    word))



(defun time-server (stream ip port peer-ip peer-port)
  "Writes a 32 Bit UNIX time in network byte order to the socket."
  (declare (ignore ip port peer-ip peer-port))
  (let ((time-bytes (word-to-bytesequence (get-universal-time))))
    (write-sequence time-bytes stream)))


  
;;; ------------

(defun add-fortunes (file)
  "Add the fortunes found in file to the fortune cookie store."
  (format t "loading fortune ~a ...~%" file)
  (with-open-file (fortune-file file :direction :input)
    (let ((line nil)
          (cookie nil)
          (fortunes 0))
      (handler-case
          (loop
           (setf line (read-line fortune-file))
           (if (< 0 (length line))
               (if (and (equal #\% (char line 0))
                        (not (null cookie)))
                   (progn
                     (sb-thread:with-mutex (*fortune-mutex* :value t :wait-p t)
                       (if (null *fortunes*)
                           (setf *fortunes* (list cookie))
                           (nconc *fortunes* (list cookie))))
                     (setf fortunes (1+ fortunes))
                     (setf cookie nil))
                   (setf cookie (concatenate 'string
                                             cookie
                                             line
                                             (string #\Newline))))
               (setf cookie (concatenate 'string
                                         cookie
                                         (string #\Newline)))))
        (end-of-file (condition)
          (declare (ignore condition))
          (return-from add-fortunes fortunes))))))



(defun filep (object)
  "Tries to guess wether the specified file system object is a file (or
a directory)"
  (let* ((check-path (probe-file object))
         (printed-path (format nil "~a" check-path)))
    (if check-path
        (if (equal #\/
                   (char printed-path (1- (length printed-path))))
            nil
            t)
        nil)))


(defun load-all-system-fortunes ()
  "Loads all the fortunes installed in the system. Looks into *fortune-path*.
Does _not_ descend recursively into the path elements."
  (let ((fortunes 0))
    (dolist (path *fortune-paths*)
      (dolist (fortune (directory path))
        (if (filep fortune)
            (setf fortunes (+ fortunes (add-fortunes fortune))))))
    (format t "loaded ~a fortune cookies.~%" fortunes))
  t)


(defun fortune-server (stream ip port peer-ip peer-port)
  "Print a randomly selected fortune cookie and exit."
  (declare (ignore ip port peer-ip peer-port))
  (sb-thread:with-mutex (*fortune-mutex* :value t :wait-p t)
    (if (< 0 (length *fortunes*))
        (format stream
                "~a"
                (nth (random
                      (length *fortunes*))
                     *fortunes*)))))

(defun hello-server (stream ip port peer-ip peer-port)
  "Say hello, print connection details and exit."
  (format stream
          "bound to: ~a:~a, connection from ~a:~a.~%"
          ip
          port
          peer-ip
          peer-port))



(defun start-simple-tcp-servers ()
  "Starts the whole load."
  ;; first, load some fortune cookies - tweak the path if needed
  (if (load-all-system-fortunes)
      (format t "Successfully loaded fortune cookies~%")
      (format t "Failed to load fortune cookies~%"))
  (if 
   (lns:register-service-handler *address*
                                 4000
                                 #'lns/simple-tcp:echo-server)
   (format t "Registered echo-server to 4000/tcp~%")
   (format t "Failed to register echo-server to 4000/tcp~%"))
  (if 
   (lns:register-service-handler *address*
                                 4001
                                 #'lns/simple-tcp:daytime-server)
   (format t "Registered daytime-server to 4001/tcp~%")
   (format t "Failed to register daytime-server to 4001/tcp~%"))
  (if 
   (lns:register-service-handler *address*
                                 4002
                                 #'lns/simple-tcp:fortune-server)
   (format t "Registered fortune-server to 4002/tcp~%")
   (format t "Failed to register fortune-server to 4002/tcp~%"))
  (if 
   (lns:register-service-handler *address*
                                 4003
                                 #'lns/simple-tcp:time-server)
   (format t "Registered time-server to 4003/tcp~%")
   (format t "Failed to register time-server to 4003/tcp~%"))
  (if 
   (lns:register-service-handler *address*
                                 4004
                                 #'lns/simple-tcp:hello-server)
   (format t "Registered hello-server to 4004/tcp~%")
   (format t "Failed to register hello-server to 4004/tcp~%")))

  

              
